%% Solar Cell Experiment - Second part
%
% This script is intended to display the figures corresponding to the
% measurement of a solar cell I-V curve. Data is to be entered in the "Data Entry" section.

% Tip : Consider the use of ... to break the entry on multiple lines !

% Clearing of the work environment
clear all
close all
clc

%% Data entry

% Voltage measured. Data between brackets is in mV
V = 1e-3*[...
    0 , 34 , 64 , 94 , 125 , ...
    154 , 184 , 213 , 241 , 269 , ...
    321 , 365 , 398 , 421 , 438 , ...
    451 , 461 , 468 , 474 , 479 , ...
    481 , 488 , 497 , 503 , 509 , ...
    514 , 519 , 524 , 530 , 535];

% Current measured. Data between brackets is in mA
I = -1e-3*[...
    299 , 299 , 298 , 298 , 298 , ...
    297 , 296 , 295 , 293 , 291 , ...
    285 , 274 , 259 , 242 , 225 , ...
    209 , 195 , 182 , 170 , 160 , ...
    155 , 139 , 119 , 102 , 86 , ...
    70 , 56 , 38 , 20 , 0];

% Algorithm parameters

n = 2.5;         % Diode ideality factor
R_step = 0.01;   % Rs incremental step in Ohm
It_max = 100;    % Maximum number of iteration
tol = 1;         % Tolerance on maximum power difference (in %)
Nb_pts = 50;     % Number of points for numerical fitting

%% Initialisation

% Some fundamental constants
h= 6.626e-34;       
c = 299792458;
q = 1.602e-19;
kB = 1.3806e-23;

% Room temperature (in K)
T = 20+273.15;

% Open circuit voltage and short circuit current
V_oc = max(V);
I_sc = -max(abs(I));

% Number of measurement points
Meas_Points = length(V);

% Maximum power point (MPP)
[~,Id_Max] = max(abs(V.*I));
V_mpp = V(Id_Max);
I_mpp = I(Id_Max);
P_max = V_mpp*I_mpp;

% Fill factor
FF = V(Id_Max)*abs(I(Id_Max))/max(V)/max(abs(I));

% Initial parameters
I_0 = -I_sc/(exp(V_oc*q/n/kB/T)-1);
Rp_min = V_mpp/(I_mpp-I_sc);
Rs = 0;
Rp = Rp_min;

% Vectors for numerical fitting
V_fit = linspace(0,V_oc,Nb_pts);
I_fit = zeros(1,Nb_pts);


%% Iteration

for m = 1:It_max
    
    % Computation of the photovoltaic current
    I_ph = -I_sc*(Rs+Rp)/Rp;
    
    % Computation of the parallel resistance
    Rp = V_mpp*(V_mpp-I_mpp*Rs)/(P_max+V_mpp*I_ph-V_mpp*I_0*exp(((V_mpp-I_mpp*Rs)*q)/(n*kB*T))+V_mpp*I_0);
    
    % Numerically solving the I-V equation
    for j = 1:Nb_pts
        
        syms x
        I_fit(j) = vpasolve(-x-I_ph+I_0*(exp((V_fit(j)-Rs*x)*q/n/kB/T)-1)+(V_fit(j)-Rs*x)/Rp,x);
        
    end
    
    % Maximum power of the found fitting curve
    P_max_fit = max(abs(V_fit.*I_fit));
    
    % Error on maximum power in %
    e = abs(P_max_fit-abs(P_max))/abs(P_max)*100;
    
    % Breaking condition
    if e<=tol
        break;
    end
    
    % Increase in Rs
    Rs = Rs+R_step;
end

%% Figures
figure(1)
hold on
plot(V,I,'linewidth',2)
plot(V,I(Id_Max).*ones(1,Meas_Points),'r--','linewidth',2)
plot(V(Id_Max).*ones(1,Meas_Points),I,'g--','linewidth',2)
plot(V_fit,I_fit,'k','linewidth',2)
grid on
xlabel('Voltage [mV]')
ylabel('Current [mA]')
legend('Data','I_{mpp}','V_{mpp}','Fitting','location','northwest')
title('I-V curve measured and corresponding modeling')

%% Display of the final results

disp(sprintf('Open circuit voltage: %d',V_oc));
disp(sprintf('Short circuit current: %d',I_sc));
disp(sprintf('Maximum power delivered: %d', P_max));
disp(sprintf('Fill factor (FF): %d', FF));
